<?php

class Application_Extension_Pdf_Page extends Zend_Pdf_Page {
	
	/**
	 * Align text at left of provided coordinates
	 */
	const TEXT_ALIGN_LEFT = 0;
	
	/**
	 * Align text at right of provided coordinates
	 */
	const TEXT_ALIGN_RIGHT = 2;
	
	/**
	 * Center-text horizontally within provided coordinates
	 */
	const TEXT_ALIGN_CENTER = 1;
	
	protected $_lineHeight = 1.15;
	
	protected $_document = null;
	
	protected $_pageNumber;
	
	/**
	 * Hauteur de l'objet précédemment peint
	 * @var float
	 */
	private $_precedent = 0;
	
	/**
	 * Position du curseur dans la page
	 * @var float
	 */
	private $_cursor = 0;
	
	public function attach(Zend_Pdf $pdf) {
		$this->_pageNumber = count($pdf->pages);
		$this->_document = $pdf;
		$pdf->pages[] = $this;
		return $this;
	}
	
	public function drawMultiLineText($text, $x, $y, $charEncoding = '', $separator = PHP_EOL) {
		foreach (explode($separator,$text) as $i => $line) {
			$this->drawText($line, $x, $y - $i * $this->_lineHeight * $this->_fontSize, $charEncoding);
		}
		return $this;
	}
	
	public function drawAlignedText($text, $x1, $y, $x2 = null, $position = self::TEXT_ALIGN_LEFT, $encoding = null, $orientation = 0, $linewrap = 0) {
		$bottom = $y; // could do the same for vertical-centering
		switch ($position) {
			case self::TEXT_ALIGN_LEFT:
				$left = $x1;
				break;
			case self::TEXT_ALIGN_RIGHT:
				if (null === $x2) {
					throw new Exception("Alignement impossible, largeur de la boîte non déductible");
				}
				$text_width = $this->getTextWidth($text, $this->getFont(), $this->getFontSize());
				$left = $x2 - $text_width;
				break;
			case self::TEXT_ALIGN_CENTER:
				if (null === $x2) {
					throw new Exception("Alignement impossible, largeur de la boîte non déductible");
				}
				$text_width = $this->getTextWidth($text, $this->getFont(), $this->getFontSize());
				$box_width = $x2 - $x1;
				$left = $x1 + ($box_width - $text_width) / 2;
				break;
			default:
				throw new Exception("Invalid position value \"$position\"");
		}
		if ($linewrap == 1 && $x2 === null) throw new Exception("Retour à la ligne impossible, largeur de la boîte non déductible");
		if ($linewrap == 1) $text = $this->balanceLines($text, $x2 - $x1);
		if ($orientation != 0) {
			$this->saveGS()->rotate($x1, $y, $orientation);
		}
		$this->drawMultiLineText($text, $left, $bottom, $encoding);
		if ($orientation != 0) {
			$this->restoreGS();
		}
		return $this;
	}
	
	public function drawMatrixcode(Application_Plugin_Matrixcode_Abstract $matrixcode, $x, $y, $width, $moduleEntier = false) {
		if ($matrixcode->getText() == "") {
			return $this;
		}
		if (!$this->_document === null) {
			throw new Exception("L'implémentation des Zend_Barcode_Renderers nécessite que cette page soit attachée à un PDF");
		}
		$matrixcode->setEccLevel('H');
		if (!$moduleEntier)
			$matrixcode->setModuleSize($width / $matrixcode->getWidth());
		else
			$matrixcode->setModuleSize(round($width / $matrixcode->getWidth()) * 144 / 203);
		if ($matrixcode->getVersion() < 12) {
			$matrixcode->setVersion(12); // si on change la version, on recalcule la largeur
			if (!$moduleEntier)
				$matrixcode->setModuleSize($width / $matrixcode->getWidth());
			else
				$matrixcode->setModuleSize(round($width / $matrixcode->getWidth()) * 144 / 203);
		}
		$matrixcode->setPadding(0);
		$renderer = new Application_Plugin_Matrixcode_Renderer_Pdf();
		$renderer->setMatrixcode($matrixcode);
		$renderer->setResource($this->_document, $this->_pageNumber);
		$renderer->setLeftOffset(intval($x));
		$renderer->setTopOffset(intval($y));
		$renderer->setSendResult(false);
		$renderer->render();
		return $this;
	}
	
	public function drawBarcode(Zend_Barcode_Object_ObjectAbstract $barcode, $x, $y, $width, $height, $fontFactor = 0.3) {
		if (!$this->_document === null) {
			throw new Exception("L'implémentation des Zend_Barcode_Renderers nécessite que cette page soit attachée à un PDF");
		}
		$renderer = new Zend_Barcode_Renderer_Pdf();
		$renderer->setBarcode($barcode);
		$barcode->setDrawText(false); // dans tous les cas, on le fait à la main
		//$barcode->setWithQuietZones(false);
		$before = $barcode->getHeight(true);
		$renderer->setModuleSize(1);
		$renderer->setModuleSize($width / $barcode->getWidth(true));
		$barcode->setBarHeight($height / $renderer->getModuleSize());
		$reserve = $before - $barcode->getHeight(true);
		
		$renderer->setResource($this->_document, $this->_pageNumber);
		$renderer->setLeftOffset(intval($x));
		$renderer->setTopOffset($this->getHeight() - $y - $barcode->getBarHeight() - $reserve);
		$renderer->draw();
		if ($fontFactor > 0) {
			$style = $this->getStyle();
			$this->setFont($this->getFont(), $barcode->getBarHeight()*$fontFactor*$renderer->getModuleSize());
			$this->drawAlignedText($barcode->getTextToDisplay(), $x, $y - $this->getFontSize(), $x + ($barcode->getWidth(true)), self::TEXT_ALIGN_CENTER, "UTF-8");
			if ($style !== null) $this->setStyle($style);
		}
		return $this;
	}
	
	private function balanceLines($text, $width) {
		$bits = explode(' ', $text);
		$balancedText = $bits[0];
		$currentLine = $bits[0];
		for ($i = 1; $i < count($bits); $i++) {
			if ($this->getTextWidth($currentLine.' '.$bits[$i], $this->getFont(), $this->getFontSize()) > $width) {
				//il faut un retour ligne
				$balancedText .= PHP_EOL.$bits[$i];
				$currentLine = $bits[$i];
			} else {
				//on peut continuer à empiler
				$balancedText .= ' '.$bits[$i];
				$currentLine .= ' '.$bits[$i];
			}
		}
		return $balancedText;
	}
	
	/**
	 * Return length of generated string in points
	 *
	 * @param string $string
	 * @param Zend_Pdf_Resource_Font $font
	 * @param int $font_size
	 * @return double
	 */
	private function getTextWidth($text, Zend_Pdf_Resource_Font $font, $font_size)
	{
		$drawing_text = @iconv('UTF-8', 'UTF-16BE', utf8_encode(strtr(utf8_decode($text), 'àáâãäçèéêëìíîïñòóôõöùúûüýÿÀÁÂÃÄÇÈÉÊËÌÍÎÏÑÒÓÔÕÖÙÚÛÜÝ€', 'aaaaaceeeeiiiinooooouuuuyyAAAAACEEEEIIIINOOOOOUUUUYE')));
		$characters    = array();
		for ($i = 0; $i < strlen($drawing_text); $i++) {
			$characters[] = (ord($drawing_text[$i++]) << 8) | ord ($drawing_text[$i]);
		}
		$glyphs        = $font->glyphNumbersForCharacters($characters);
		$widths        = $font->widthsForGlyphs($glyphs);
		$text_width   = (array_sum($widths) / $font->getUnitsPerEm()) * $font_size;
		return $text_width;
	}
	
	public function getLineHeight() {
		return $this->_lineHeight;
	}

	public function setLineHeight($_lineHeight) {
		$this->_lineHeight = $_lineHeight;
		return $this;
	}
	
	public function showGrid($step = 5) {
		$style = $this->getStyle();
		$this->setFont(Zend_Pdf_Font::fontWithName(Zend_Pdf_Font::FONT_HELVETICA), ceil($step / 2));
		$this->setFillColor(Zend_Pdf_Color_Html::namedColor('red'));
		for ($j = $step; $j < $this->getHeight(); $j += $step) {
			$this->drawLine(0, $j, $this->getWidth(), $j);
			$this->drawText($j, 0, $j);
		}
		for ($i = $step; $i < $this->getWidth(); $i += $step) {
			$this->drawLine($i, 0, $i, $this->getHeight());
			$this->drawText($i, $i, 0);
		}
		$this->setStyle($style);
	}
	public function getCursor() {
		return $this->_cursor;
	}

	public function setCursor($_cursor) {
		$this->_cursor += $_cursor;
		return $this;
	}
	public function getPrecedent() {
		return $this->_precedent;
	}

	public function setPrecedent($_precedent) {
		$this->_precedent = $_precedent;
		return $this;
	}



}