<?php


if (!defined("PROPRIETAIRE") && Zend_Auth::getInstance()->hasIdentity()) define("PROPRIETAIRE", Zend_Auth::getInstance()->getStorage()->read()->id);

class Application_Model_AdresseMapper
{
	protected $_dbTable;
	
	public function setDbTable($dbTable)
    {
        if (is_string($dbTable)) {
            $dbTable = new $dbTable();
        }
        if (!$dbTable instanceof Zend_Db_Table_Abstract) {
            throw new Exception('Invalid table data gateway provided');
        }
        $this->_dbTable = $dbTable;
        return $this;
    }
 
    public function getDbTable()
    {
        if (null === $this->_dbTable) {
            $this->setDbTable('Application_Model_DbTable_Adresse');
        }
        return $this->_dbTable;
    }
    /**
     * enregistre une adresse dans la base
     * @param Application_Model_Adresse $adresse
     */
    public function save(Application_Model_Adresse $adresse) {
      $data = array(
        ":ref"    => $adresse->getLibelle(),
        ":nom"    => $adresse->getNom(),
        ":ligne1" => mb_substr($adresse->getLigne1(),0,38,'UTF-8'),
        ":ligne2" => mb_substr($adresse->getLigne2(),0,38,'UTF-8'),
        ":ligne3" => mb_substr($adresse->getLigne3(),0,38,'UTF-8'),
        ":ligne4" => mb_substr($adresse->getLigne4(),0,38,'UTF-8'),
        ":code"   => $adresse->getCodePostal(),
        ":ville"  => $adresse->getVille(),
        ":pays"   => $adresse->getPays(),
        ":tag"    => $adresse->getTag()
      );
    	if (null === ($id = $adresse->getId())) {  	
    		$sql = "INSERT INTO adresse(idtiers, libelle, nom, ligne1, ligne2, ligne3, ligne4, codepostal, ville, pays, tag) VALUES (:idt,:ref,:nom,:ligne1,COALESCE(:ligne2,''),COALESCE(:ligne3,''),COALESCE(:ligne4,''),:code,:ville,:pays,:tag);";
    		$data[':idt'] = $adresse->getTiers();
    	} else {
    		$sql = "UPDATE adresse SET (libelle, nom, ligne1, ligne2, ligne3, ligne4, codepostal, ville, pays, tag) = (:ref,:nom,:ligne1,COALESCE(:ligne2,''),COALESCE(:ligne3,''),COALESCE(:ligne4,''),:code,:ville,:pays,:tag) WHERE id = ".$id.";";
    	}
    	$stmt = new Zend_Db_Statement_Pdo($this->getDbTable()->getAdapter(), $sql);
    	$stmt->execute($data);
    }
    
    /**
     * récupère toutes les adresses du tiers dont l'id est $id.
     * Si $id est null, tout le carnet est récupéré.
     * @param int $id
     * @return array[Application_Model_Adresse] $entries
     */
    public function fetchAll($id = null) {
      $resultSet = $this->getDbTable()->getAdapter()->select()->from('adresse')->where('idtiers = ?', $id)->query()->fetchAll(Zend_Db::FETCH_OBJ);
      $entries = array();
      foreach ($resultSet as $row) {
        $entry = new Application_Model_Adresse();
        $entry->setId($row->id)
            ->setTiers($row->idtiers)
            ->setLibelle($row->libelle)
            ->setLigne1($row->ligne1)
            ->setLigne2($row->ligne2)
            ->setLigne3($row->ligne3)
            ->setLigne4($row->ligne4)
            ->setCodePostal($row->codepostal)
            ->setVille($row->ville)
            ->setPays($row->pays)
            ->setTag($row->tag);
        $entries[] = $entry;
      }
      return $entries;
    }
    
    /**
     * récupère les adresses de l'expéditeur courant en plaçant celle par défaut en premier
     * return array[Application_Model_Adresse]
     */
    public function fetchAllAndDefault() {
    	$stmt = new Zend_Db_Statement_Pdo($this->getDbTable()->getAdapter(),
    			"SELECT adresse.* FROM adresse JOIN expediteur ON adresse.idtiers = expediteur.idtiers
    			WHERE adresse.idtiers = :id ORDER BY adresse.id = expediteur.idadressedefaut DESC;");
    	$stmt->execute(array("id" => PROPRIETAIRE));
    	$resultSet = $stmt->fetchAll(Zend_Db::FETCH_ASSOC);
    	$entries = array();
    	foreach ($resultSet as $row) {
    		$entry = new Application_Model_Adresse($row);
    		$entry->setTiers($row['idtiers']);
    		$entries[] = $entry;
    	}
    	return $entries;
    }
    
    /**
     * 
     */
    public function fetchAllObj($id = null) {
    	$stmt = new Zend_Db_Statement_Pdo($this->getDbTable()->getAdapter(), "SELECT * FROM vueadresse WHERE idtiers = :id;");
    	$stmt->execute(array(":id" => $id));
    	$resultSet = $stmt->fetchAll(Zend_Db::FETCH_ASSOC);
    	$entries = array();
    	foreach ($resultSet as $row) {
    		$entry = new Application_Model_ObjAdresse($row);
    		$entries[] = $entry;
    	}
    	return $entries;
    }
    
    /**
     * Récupère le carnet entier sous forme de tableau bidimensionnel contenant un objet Tiers et un objet Adresse par ligne
     * @return array $entries au format (t=>Tiers,a=>Adresse)
     */
    public function fetchAllForExport() {
    	$resultSet = $this->getDbTable()->getAdapter()->select()->from('vueexportcarnet')->where("idexpediteur = ?", PROPRIETAIRE)->query()->fetchAll(Zend_Db::FETCH_ASSOC);
    	$entries = array();
    	foreach($resultSet as $row) {
    		$type = "Application_Model_".ucfirst($row['type']);
    		$entries[] = array("t" => new $type($row, true), "a" => new Application_Model_Adresse($row));
    	}
    	return $entries;
    }
    
    /**
     * Récupère le carnet entier sous forme de tableau associatif.
     * Originellement destiné à la saisie LV
     * @return array $entries au format id => tiers-adresse
     */
    public function fetchAllAsList() {
    	$sql = "SELECT adresse.id, adresse.libelle, tiers.reference FROM adresse JOIN tiers ON adresse.idtiers = tiers.id WHERE tiers.idexpediteur = :prop AND tiers.id <> tiers.idexpediteur;";
    	$stmt = new Zend_Db_Statement_Pdo($this->getDbTable()->getAdapter(), $sql);
    	$stmt->execute(array("prop" => PROPRIETAIRE));
    	$resultSet = $stmt->fetchAll(Zend_Db::FETCH_OBJ);
    	$entries = array();
    	foreach ($resultSet as $result) {
    		$entries[$result->id] = $result->reference."-".$result->libelle;
    	}
    	return $entries;
    }
    
    /**
     * Trouve l'adresse correspondant à $id et la retourne dans $adresse
     * @param int $id
     * @param Application_Model_Adresse &$adresse
     */
    public function find($id, Application_Model_Adresse $adresse) {
    	$row = $this->getDbTable()->getAdapter()->select()->from('adresse')->where("id = ?", $id)->query()->fetch(Zend_Db::FETCH_OBJ);
    	if (0 == count($row)) {
    		return;
    	}
    	$adresse->setId($row->id)
    			->setTiers($row->idtiers)
    			->setLibelle($row->libelle)
                ->setNom($row->nom)
    			->setLigne1($row->ligne1)
    			->setLigne2($row->ligne2)
				->setLigne3($row->ligne3)
    			->setLigne4($row->ligne4)
    			->setCodePostal($row->codepostal)
    			->setVille($row->ville)
    			->setTag($row->tag)
    			->setPays($row->pays);
    }
    
    /**
     * Trouve l'adresse correspondant à $id et la retourne dans un format compatible avec les adresses statiques
     * @param int $id
     * @param Application_Model_ObjAdresse $objadresse
     */
    public function findObj($id, Application_Model_ObjAdresse $objadresse) {
    	$row = $this->getDbTable()->getAdapter()->select()->from('vueadresse')->where("id = ?", $id)->query()->fetch(Zend_Db::FETCH_OBJ);
    	if (0 == count($row)) {
    		return;
    	}
    	$objadresse->setId($row->id)
		    	   ->setNomComplet($row->nomcomplet)
		    	   ->setLigne1($row->ligne1)
    			   ->setLigne2($row->ligne2)
				   ->setLigne3($row->ligne3)
    			   ->setLigne4($row->ligne4)
		    	   ->setCodePostal($row->codepostal)
		    	   ->setVille($row->ville)
    			   ->setTag($row->tag)
    			   ->setTelephone($row->telephone)
    			   ->setLibelle($row->libelle)
    			   ->setPays($row->pays)
    			   ->setReference($row->reference);
    }
    
    /**
     * 
     * @param int $id
     * @return assoc $row
     */
    public function findFullDetails($id) {
    	return $this->getDbTable()->getAdapter()->select()->from('vuecarnet')->where("idadresse = ?", $id)->query()->fetch(Zend_Db::FETCH_ASSOC);
    }
    
    /**
     * Recherche une adresse et retourne son id si elle existe.
     * Cette fonction ne devrait être utilisée que si le tiers est connu, car rechercher une adresse quel que soit le tiers n'a pas de sens, deux entreprises peuvent loger à la même adresse
     * @param Application_Model_Adresse $adresse
     * @return $id l'id de l'adresse si elle existe, null sinon
     */
    public function exists(Application_Model_Adresse $adresse) {
    	$select = $this->getDbTable()->getAdapter()->select()
    											->from('adresse')
    											->where("idtiers = ?", $adresse->getTiers())
    											->where('ligne1 = ?',$adresse->getLigne1())
												->where('ligne2 = ?',$adresse->getLigne2())
												->where('ligne3 = ?',$adresse->getLigne3())
												->where('ligne4 = ?',$adresse->getLigne4())
												->where('codepostal = ?',$adresse->getCodePostal())
												->where('ville = ?',$adresse->getVille());
		$ligne2 = $adresse->getLigne2();
		if (!empty($ligne2)) {
			$select->where('ligne2 = ?',$adresse->getLigne2()); // lève une exception si la ligne est vide, d'où la vérification
		}
		$ligne3 = $adresse->getLigne3();
		if (!empty($ligne3)) {
			$select->where('ligne3 = ?',$adresse->getLigne3()); // lève une exception si la ligne est vide, d'où la vérification
		}
		$ligne4 = $adresse->getLigne4();
		if (!empty($ligne4)) {
			$select->where('ligne4 = ?',$adresse->getLigne4()); // lève une exception si la ligne est vide, d'où la vérification
		}
		$row = $select->query()->fetch(Zend_Db::FETCH_OBJ);
		if ($row === false) {
			return null;
		} else {
			return $row->id;
		}
    }
    
    /**
     * 
     * @param int $id
     */
    public function count($id) {
    	$stmt = new Zend_Db_Statement_Pdo($this->getDbTable()->getAdapter(), "SELECT idtiers, count(*) AS nb FROM adresse WHERE idtiers = (SELECT idtiers FROM adresse WHERE id = :id) GROUP BY idtiers;");
    	$stmt->execute(array(":id" => $id));
    	return $stmt->fetch(Zend_Db::FETCH_OBJ);
    }
    
    /**
     * Supprime l'adresse correspondant à $id
     * @param int $id
     */
    public function delete($id) {
    	$o = $this->count($id);
    	if ($o->nb == 1) return $o->idtiers;
    	if ($this->isDefault(PROPRIETAIRE, $id)) return;
    	$idtiers = $this->getDbTable()->getAdapter()->select()->from("adresse","idtiers")->where("id = ?", $id)->query()->fetch(Zend_Db::FETCH_OBJ)->idtiers;
    	$stmt = new Zend_Db_Statement_Pdo($this->getDbTable()->getAdapter(), "DELETE FROM adresse WHERE id = :id;");
    	$stmt->execute(array(":id" => $id));
    	return $idtiers;
    }
    
    /**
     * Détermine si l'adresse correspondant à $adresse est celle par défaut pour l'utilisateur correspondant à $user
     * @param int $user
     * @param int $adresse
     * @return boolean
     */
    public function isDefault($user, $adresse) {
    	$row = $this->getDbTable()->getAdapter()->select()->from('expediteur')->where("idtiers = ?", $user)->where("idadressedefaut = ?", $adresse)->query()->fetch(Zend_Db::FETCH_OBJ);
    	// une requête ne sélectionnant aucune ligne fetchera FALSE
    	if ($row === false) { // et count(FALSE) renvoie 1, il faut donc tester directement $row 
    		return false;
    	}
    	return true;
    }
    
    public function getDefault($user) {
    	$row = $this->getDbTable()->getAdapter()->select()->from('expediteur','idadressedefaut')->where("idtiers = ?", $user)->query()->fetch(Zend_Db::FETCH_OBJ);
    	return $row->idadressedefaut;
    }
    
    public function setDefault($user, $adresse) {
    	$stmt = new Zend_Db_Statement_Pdo($this->getDbTable()->getAdapter(), "UPDATE expediteur SET idadressedefaut = :ad WHERE idtiers = :id");
    	$stmt->execute(array(":id" => $user, ":ad" => $adresse));
    }
}

