<?php

class Application_Model_VilleMapper
{
	/*
	 * Les codes postaux peuvent représenter 1 à n villes
	 * Les villes peuvent représenter 1 à n codes postaux
	 */
	private $_codes;

	private static $_pays = null;

	private static $_regexp_cp = null;
	
	/**
	 * Constructeur du mapper de villes d'après le fichier fr.villes.origine
	 */
	public function __construct($pays) {
		if (is_null($pays)) return;
		$fileName = '../application/models/Data/'.$pays.'.villes';
		if (!file_exists($fileName)) {
			Zend_Registry::get('logger')->info("Can't find file \"$fileName\" in order to get city list.");
			return;
		}
		$fichier = fopen($fileName,'r');
		$this->_codes = [];
		fgets($fichier); // on passe le nom du pays
		$rx = substr(fgets($fichier), 0, -2); // on récupère la regexp pour vérifier que tous les codes postaux la satisfont
		while ($ligne = fgets($fichier)) {
			list($codepostal, $ville) = explode(" ",substr($ligne, 0, -2)); // retire les \r\n en fin de ligne
			if (preg_match($rx, $codepostal) === 0) trigger_error('Le code '.$codepostal.' pour la ville '.$ville.' ('.$pays.') ne satisfait pas l\'expression reguliere renseignee', E_USER_WARNING);
			if (!isset($this->_codes[$ville])) $this->_codes[$ville] = array();
			$this->_codes[$ville][] = $codepostal;
		}
		fclose($fichier);
	}

	private static function browsePays() {
		if (!is_null(self::$_pays)) return;
		foreach (glob('../application/models/Data/??.villes') as $fichier_villes) {
			$iso = basename($fichier_villes, ".villes");
			$fh = fopen($fichier_villes, 'r');
			self::$_pays[$iso] = substr(fgets($fh), 0, -2);
			self::$_regexp_cp[$iso] = substr(fgets($fh), 0, -2);
		}
		fclose($fh);
	}

	public static function getPaysDisponibles() {
		self::browsePays();
		return self::$_pays;
	}

	public static function getRegexpCp($pays) {
		self::browsePays();
		return self::$_regexp_cp[$pays];
	}
	
	/**
	 * Utilisé pour la recherche manuelle de ville
	 * renvoie uniquement le premier code postal d'une ville donnée
	 * @param string $ville la ville recherchée
	 * @return array $code le code postal de la ville donnée dans un tableau associatif, à l'indice $ville
	 */
	public function findCode($ville) {
		$ville = strtr(strtoupper($ville)," '","--");
		if (isset($this->_codes[$ville])) {
			return array($ville => $this->_codes[$ville][0]);
		}
		return $this->findCode($this->findClosest($ville));
	}
	
	/**
	 * Retourne les villes correspondant au code postal fourni
	 * @param string $codepostal
	 * @return array $villes|null
	 */
	public function findVille($codepostal) {
		if(count($this->_codes) < 1) return null;
		$matches = array();
		foreach($this->_codes as $ville => $codes) {
			foreach ($codes as $code) {
				if ($code === $codepostal) {
					$matches[] = $ville;
				}
			}
		}
		if (count($matches) > 0) {
			return $matches;
		}
		return null;
	}
	
	/**
	 * retourne la ville existante la plus proche de celle recherchée
	 * @param string $ville_cherchee
	 * @return string|null
	 */
	public function findClosest($ville_cherchee) {
		$best = "";
		$smallest = 255;
		foreach($this->_codes as $ville => $code) {
			if (($current = levenshtein($ville, $ville_cherchee)) < $smallest) {
				$best = $ville;
				$smallest = $current;
			}
		}
		if ($smallest < 255) {
			return $best;
		}
		return null;
	}
}

