<?php

class Application_Plugin_Pdf_XMLParser {
	
	/**
	 * Tableau des noeuds du fichier d'état.
	 * @var tableau de SimpleXMLElements
	 */
	private $_ordres;
	
	/**
	 * Fichier pdf dans lequel écrire les données.
	 * @var Zend_Pdf
	 */
	private $_pdf;
	
	/**
	 * Ensemble de formats standards pour la construction rapide de page.
	 * @var tableau de formats standards
	 */
	private $_formats = array(
		"A7" => "297:210:",
		"A6" => "420:297:",
		"A5" => "594:420:",
		"A4" => "594:840:"
	);
	
	/**
	 * Nombre de répétitions des actions pour impression multiple (HOSPITON)
	 * @var int
	 */
	private $_repetitions = 0;
	
	/**
	 * Décalage entre chaque répétition pour impression multiple (HOSPITON)
	 * @var float
	 */
	private $_decalage = 0;
	
	/**
	 * Contient les fontes utilisées au long de l'état.
	 * L'indice est de la forme FontNameFontStyle.
	 * Non garanti pour d'autres polices qu'Arial.
	 * @var tableau de Zend_Pdf_Font
	 */
	private $_fonts = array();
	
	/**
	 * Contient les couleurs utilisées au long de l'état.
	 * @var tableau de Zend_Pdf_Color
	 */
	private $_colors = array();
	
	/**
	 * Contient les images de l'état
	 * @var tableau de Zend_Pdf_Resource_Image
	 */
	private $_images = array();
	
	/**
	 * Contient les objets métiers dans lesquels les données sont lues.
	 * @var tableau associations d'objets métiers/tableaux additionnels
	 */
	private $_objects = array();
	
	/**
	 * Zone où dessiner
	 * @var Zend_Pdf_Canvas
	 */
	private $_zone = null;
	
	/**
	 * Espace disponible sur une page, pour le calcul du nombre de pages
	 * @var float
	 */
	private $_espace = 0.0;
	
	/**
	 * Nombre de pages à imprimer dans certains états
	 * @var int
	 */
	private $_nbpages = 0;
	
	/**
	 * Constante de conversion des millimètres en points.
	 * @var const
	 */
	const TAUX_MM_PT = 2.8346;
	
	/**
	 * Constante de conversion des degrés en radians
	 * @var const
	 */
	const TAUX_DEG_RAD = 0.01745;
	
	/**
	 * Constructeur de l'interpréteur.
	 * @param string $filename
	 * @throws Exception
	 */
	public function __construct($filename) {
		if (false !== ($this->_ordres = simplexml_load_file($filename))) return;
		throw new Exception("Le fichier $filename n'a pas pu être ouvert ou n'est pas un fichier valide");
	}
	
	/**
	 * Lance l'impression d'un objet sur la première page disponible.
	 * @param Zend_Pdf $pdf le document dans lequel imprimer
	 * @param array $specs les objets métiers lus
	 */
	public function exec(Zend_Pdf $pdf, array $specs, $bande = "Bande0") {
		foreach($specs as $index => $obj) {
			if (is_array($obj)) {
				$specs[$index] = (object)($obj); // les tableaux sont castés en objets pour la lecture par opérateur flèche
			}
		}
		$this->_objects = $specs;
		$this->_pdf = $pdf;
		$this->loadFonts();
		$this->loadColors();
		$this->loadImages();
		if (isset($this->_ordres->Parametres)) {
			$this->checkParams($this->_ordres->Parametres);
		}
		$piedDePage = (float) $this->_ordres->BandePiedPage->Params->Hauteur*self::TAUX_MM_PT;
		$margeBasse = (float) $this->_ordres->Parametres->MargeBas*self::TAUX_MM_PT;
		$hauteur = (float) ($this->_ordres->Bande0->Params->Hauteur + $this->_ordres->Bande0->Params->Repetitions * $this->_ordres->Bande0->Params->Hauteur) * self::TAUX_MM_PT;
		// nb : la hauteur inclut les répétitions
		if (count($pdf->pages) > 0) { // s'il y a déjà des pages
			$page = $pdf->pages[count($pdf->pages)-1];
			if ($page->getHeight() - $page->getCursor() < $hauteur + $margeBasse + $piedDePage) { // s'il ne reste pas assez de place pour peindre l'objet
				$page = $this->newPage();
			}
		} else { // sinon, on crée une première page
			$page = $this->newPage();
		}
		// décalage du système de coordonnées en fonction de l'objet peint précédemment
		$page->setCursor($hauteur)->translate(0, -$page->getPrecedent())->setPrecedent($hauteur);
		// on parcourt la bande
		foreach($this->_ordres->$bande as $ordre) $this->action($ordre);
		//$this->debug();
	}
	
	public function newPage(Zend_Pdf $pdf = null) {
		if (!is_null($pdf)) {
			$this->_pdf = $pdf;
			$this->loadFonts();
			$this->loadColors();
			$this->loadImages();
		}
		if (isset($this->_formats[(string) $this->_ordres->Parametres->Format])) {
			$page = new Application_Extension_Pdf_Page($this->_formats[(string) $this->_ordres->Parametres->Format]);
		} else {
			$page = new Application_Extension_Pdf_Page((float)$this->_ordres->Parametres->Largeur*self::TAUX_MM_PT.":".(float)$this->_ordres->Parametres->Hauteur*self::TAUX_MM_PT.":");
		}
		// en premier lieu, on décale le système de coordonnées de la valeur de marge gauche
		$page->attach($this->_pdf)->translate($this->_ordres->Parametres->MargeGauche*self::TAUX_MM_PT, -$page->getHeight() + $this->_ordres->Parametres->MargeBas*self::TAUX_MM_PT + $this->_ordres->BandePiedPage->Params->Hauteur*self::TAUX_MM_PT);
		if (count($this->_pdf->pages) == 1 && isset($this->_objects["adds"]->nbLv)) { // le calcul de la hauteur disponible et du nb de pages ne se fait qu'une seule fois
			$entete = ($this->_ordres->Parametres->MargeHaut + $this->_ordres->BandeEntete->Params->Hauteur + $this->_ordres->BandeTitre->Params->Hauteur) * self::TAUX_MM_PT;
			$bande1 = (float) $this->_ordres->Bande1->Params->Hauteur*self::TAUX_MM_PT;
			$this->_espace = $page->getHeight() - $entete - $bande1 - $this->_ordres->Parametres->MargeBas*self::TAUX_MM_PT - $this->_ordres->BandePiedPage->Params->Hauteur*self::TAUX_MM_PT;
			$nbLvParPage = floor($this->_espace / (($this->_ordres->Bande0->Params->Hauteur + $this->_ordres->Bande0->Params->Repetitions * $this->_ordres->Bande0->Params->Hauteur) * self::TAUX_MM_PT));
			$this->_nbpages = ceil($this->_objects["adds"]->nbLv / $nbLvParPage);
			
		}
		
		// on peint ensuite la bande pied de page
		foreach($this->_ordres->BandePiedPage as $ordre) $this->action($ordre);
		// on replace le système de coordonnées en haut de la page (en conservant la marge gauche)
		$page->translate(0, $page->getHeight() - $this->_ordres->Parametres->MargeBas*self::TAUX_MM_PT - $this->_ordres->BandePiedPage->Params->Hauteur*self::TAUX_MM_PT);
		// après cela, on décale le système de coordonnées de la valeur de marge haute
		$page->setCursor($this->_ordres->Parametres->MargeHaut*self::TAUX_MM_PT)->translate(0, -$this->_ordres->Parametres->MargeHaut*self::TAUX_MM_PT);

		// pour chaque page, on peint le titre
		foreach($this->_ordres->BandeTitre as $ordre) $this->action($ordre);
		$page->setCursor($this->_ordres->BandeTitre->Params->Hauteur*self::TAUX_MM_PT)->translate(0, -$this->_ordres->BandeTitre->Params->Hauteur*self::TAUX_MM_PT);
		
		// sur la première page, on peint l'entête
		if (count($this->_pdf->pages) == 1) {
			foreach($this->_ordres->BandeEntete as $ordre) $this->action($ordre);
			$page->setCursor($this->_ordres->BandeEntete->Params->Hauteur*self::TAUX_MM_PT)->translate(0, -$this->_ordres->BandeEntete->Params->Hauteur*self::TAUX_MM_PT);
		}
		return $page;
	}
	
	/**
	 * Fonction fourre-tout permettant d'imprimer des informations complémentaires sur une nouvelle page.
	 * Décommenter la ligne correspondante dans la méthode exec() pour activer la fonction
	 */
	private function debug() {
		$page = new Application_Extension_Pdf_Page("297:210:");
		$page->setFont(Zend_Pdf_Font::fontWithName(Zend_Pdf_Font::FONT_HELVETICA), 5);
		$page->attach($this->_pdf);
		$page->setFillColor(new Zend_Pdf_Color_Rgb(0.5, 0.5, 0.5));
		$page->drawRectangle(0, 0, 297, 210);
		$i = 0;
		foreach($this->_colors as $val => $obj) {
			$page->setFillColor($obj);
			$page->drawText($val, 5, 200-$i);
			$i += 7;
		}
		//$page->drawMultiLineText(print_r($this->_colors, true), 5, 205, "UTF-8", "\n");
	}
	
	/**
	 * Vérifie que les paramètres fournis sont compatibles avec ce parser
	 * @param SimpleXMLElement $params
	 * @throws Exception
	 */
	private function checkParams(SimpleXMLElement $params) {
	}
	
	
	/**
	 * Charge les images en mémoire (notamment pour les répétitions de bandes)
	 */
	private function loadImages() {
		foreach ($this->_ordres as $bande) {
			foreach ($bande as $objet) {
				if ($objet->getName() != "Params") {
					if ($objet->Type == "Image") {
						if (!isset($this->_images[(string) $objet->FichierImage])) {
							$this->_images[(string) $objet->FichierImage] = Zend_Pdf_Image::imageWithPath(APPLICATION_PATH."/layouts/etats/img/".$objet->FichierImage);
						}
					}
				}
			}
		}
	}
	
	/**
	 * Charge les fonts en mémoire
	 */
	private function loadFonts() {
		foreach ($this->_ordres as $bande) {
			foreach ($bande as $objet) {
				if ($objet->getName() != "Params") {
					if (isset($objet->FontName)) {
						if (function() {
							foreach ($this->_fonts as $i => $font) {
								if ($i == $objet->FontName.$objet->FontStyle) {
									return false;
								}
							}
							return true;
						}) {
							switch ($objet->FontStyle) {
								case 1:
									$font = Zend_Pdf_Font::FONT_HELVETICA_BOLD;
									break;
								case 4:
									$font = Zend_Pdf_Font::FONT_HELVETICA_ITALIC;
									break;
								case 5:
									$font = Zend_Pdf_Font::FONT_HELVETICA_BOLD_ITALIC;
									break;
								case 0:
								default:
									$font = Zend_Pdf_Font::FONT_HELVETICA;
									break;
							}
							$this->_fonts[$objet->FontName.$objet->FontStyle] = Zend_Pdf_Font::fontWithName($font, Zend_Pdf_Font::EMBED_DONT_EMBED);
						}
					}
				}
			}
		}
	}
	
	/**
	 * Charge les couleurs en mémoire
	 */
	private function loadColors() {
		foreach ($this->_ordres as $bande) {
			foreach ($bande as $objet) {
				if ($objet->getName() != "Params") {
					if (isset($objet->TextColor)) {
						if (function() {
							foreach ($this->_colors as $i => $color) {
								if ($i == $objet->TextColor) {
									return false;
								}
							}
							return true;
						}) {
							$this->_colors[(string) $objet->TextColor] = Zend_Pdf_Color_Html::color("#".str_pad(base_convert($objet->TextColor % 16777216, 10, 16),6,"0",STR_PAD_LEFT));
						}
					}
					if (isset($objet->BackColor)) {
						if (function() {
							foreach ($this->_colors as $i => $color) {
								if ($i == $objet->BackColor) {
									return false;
								}
							}
							return true;
						}) {
							$this->_colors[(string) $objet->BackColor] = Zend_Pdf_Color_Html::color("#".str_pad(base_convert($objet->BackColor % 16777216, 10, 16),6,"0",STR_PAD_LEFT));
						}
					}
					if (isset($objet->LineColor)) {
						if (function() {
							foreach ($this->_colors as $i => $color) {
								if ($i == $objet->LineColor) {
									return false;
								}
							}
							return true;
						}) {
							$this->_colors[(string) $objet->LineColor] = Zend_Pdf_Color_Html::color("#".str_pad(base_convert($objet->LineColor % 16777216, 10, 16),6,"0",STR_PAD_LEFT));
						}
					}
					if (isset($objet->Couleur)) {
						if (function() {
							foreach ($this->_colors as $i => $color) {
								if ($i == $objet->Couleur) {
									return false;
								}
							}
							return true;
						}) {
							$this->_colors[(string) $objet->Couleur] = Zend_Pdf_Color_Html::color("#".str_pad(base_convert($objet->Couleur % 16777216, 10, 16),6,"0",STR_PAD_LEFT));
						}
					}
				}
			}
		}
	}
	
	/**
	 * Exécute l'action correspondant au noeud en paramètre
	 * @param SimpleXMLElement $ordre
	 */
	private function action(SimpleXMLElement $ordre) {
		if (substr($ordre->getName(), 0, 5) === "Bande") { // recursion dans les bandes
			$bande = $ordre;
			if (isset($bande->Params)) {
				$params = $bande->Params;
				if (isset($params->Repetitions)) {
					$this->_repetitions = $params->Repetitions + 1;
					if (!isset($params->Hauteur)) throw new Exception("Les répétitions ne peuvent se faire sans préciser de décalage");
					$this->_decalage = $params->Hauteur;
				}
			}
			foreach($bande as $ordre) {
				$this->action($ordre);
			}
			$this->_repetitions = 0; // répétitions et décalage ne s'appliquent qu'à la bande courante, on les réinitialise donc
			$this->_decalage = 0;
		} elseif ($ordre->getName() != "Params" && $ordre->getName() != "Parametres") {
			$page = $this->_pdf->pages[count($this->_pdf->pages) -1];
			if (isset($ordre->BackColor) && $ordre->Transparent == 0) {
				if ($ordre->BackColor != 16777215) {
					$page->setFillColor($this->_colors[$ordre->BackColor % 16777216]);
					$page->drawRectangle($ordre->Left*self::TAUX_MM_PT, $page->getHeight()-$ordre->Top*self::TAUX_MM_PT, ($ordre->Width + $ordre->Left)*self::TAUX_MM_PT, $page->getHeight()-($ordre->Top + $ordre->Height)*self::TAUX_MM_PT, 1);
				}
			}
			if (isset($ordre->FontName) && isset($ordre->TextColor)) {
				$page->setFont($this->_fonts[$ordre->FontName.$ordre->FontStyle], (string) $ordre->FontSize);
				$page->setFillColor($this->_colors[$ordre->TextColor % 16777216]);
			}
			for ($i = 0; $i < $this->_repetitions; $i++) {
				$d = $i * $this->_decalage * self::TAUX_MM_PT;
				if ($ordre->Clipped == 1) {
					$page->saveGS();
					$page->clipRectangle($ordre->Left*self::TAUX_MM_PT, $page->getHeight()-$ordre->Top*self::TAUX_MM_PT-$d, ($ordre->Width + $ordre->Left)*self::TAUX_MM_PT, $page->getHeight()-$d-($ordre->Top + $ordre->Height)*self::TAUX_MM_PT);
				}
				switch ($ordre->Type) {
					case "Fixed":
						$page->drawAlignedText($ordre->Prefixe.$ordre->Texte.$ordre->Suffixe, $ordre->Left*self::TAUX_MM_PT, $page->getHeight()-$ordre->Top*self::TAUX_MM_PT-$d-$page->getFontSize(), ($ordre->Left + $ordre->Width)*self::TAUX_MM_PT, $ordre->Alignement, "UTF-8", $ordre->Orientation*self::TAUX_DEG_RAD, $ordre->WordWrap);
						break;
					case "NumPage":
						$page->drawAlignedText(count($this->_pdf->pages), $ordre->Left*self::TAUX_MM_PT, $page->getHeight()-$ordre->Top*self::TAUX_MM_PT-$d-$page->getFontSize(), ($ordre->Left + $ordre->Width)*self::TAUX_MM_PT, $ordre->Alignement, "UTF-8", $ordre->Orientation*self::TAUX_DEG_RAD);
						break;
					case "NbrPages":
						$page->drawAlignedText($this->_nbpages, $ordre->Left*self::TAUX_MM_PT, $page->getHeight()-$ordre->Top*self::TAUX_MM_PT-$d-$page->getFontSize(), ($ordre->Left + $ordre->Width)*self::TAUX_MM_PT, $ordre->Alignement, "UTF-8", $ordre->Orientation*self::TAUX_DEG_RAD);
						break;
					case "Accumulate":
						$champ = $ordre->FormName;
						$page->drawAlignedText($ordre->Prefixe.$this->_objects['adds']->$champ.$ordre->Suffixe, $ordre->Left*self::TAUX_MM_PT, $page->getHeight()-$ordre->Top*self::TAUX_MM_PT-$d-$page->getFontSize(), ($ordre->Left + $ordre->Width)*self::TAUX_MM_PT, $ordre->Alignement, "UTF-8", $ordre->Orientation*self::TAUX_DEG_RAD);
						break;
					case "Counter":
						$page->drawAlignedText($ordre->Prefixe.$this->_objects['adds']->Counter.$ordre->Suffixe, $ordre->Left*self::TAUX_MM_PT, $page->getHeight()-$ordre->Top*self::TAUX_MM_PT-$d-$page->getFontSize(), ($ordre->Left + $ordre->Width)*self::TAUX_MM_PT, $ordre->Alignement, "UTF-8", $ordre->Orientation*self::TAUX_DEG_RAD);
						break;
					case "ObjText":
						$page->drawAlignedText($ordre->Prefixe.$this->translate(isset($ordre->NomTable)?strtolower($ordre->NomTable).".".$ordre->NomChamp:$ordre->NomChamp).$ordre->Suffixe, $ordre->Left*self::TAUX_MM_PT, $page->getHeight()-$ordre->Top*self::TAUX_MM_PT-$d-$page->getFontSize(), ($ordre->Left + $ordre->Width)*self::TAUX_MM_PT, $ordre->Alignement, "UTF-8", $ordre->Orientation*self::TAUX_DEG_RAD, $ordre->WordWrap);
						break;
					case "GroupBox":
						$page->setLineWidth(0.3);
						$page->drawRectangle(($ordre->Left)*self::TAUX_MM_PT-1, $page->getHeight()-$ordre->Top*self::TAUX_MM_PT-$d+1, ($ordre->Width + $ordre->Left)*self::TAUX_MM_PT+1, $page->getHeight()-$d-($ordre->Top + $ordre->Height)*self::TAUX_MM_PT-1, 0);
						$page->drawAlignedText($ordre->Prefixe.$ordre->Texte.$ordre->Suffixe, $ordre->Left*self::TAUX_MM_PT+.5, $page->getHeight()-$ordre->Top*self::TAUX_MM_PT-$d-$page->getFontSize(), ($ordre->Left + $ordre->Width - .5)*self::TAUX_MM_PT, $ordre->Alignement, "UTF-8");
						break;
					case "DateText":
						$page->drawAlignedText($ordre->Prefixe.date($this->formatDate($ordre->TypeDate)).$ordre->Suffixe, $ordre->Left*self::TAUX_MM_PT, $page->getHeight()-$ordre->Top*self::TAUX_MM_PT-$d-$page->getFontSize(), ($ordre->Left + $ordre->Width)*self::TAUX_MM_PT, $ordre->Alignement, "UTF-8");
						break;
					case "MatrixCode":
						$page->drawMatrixcode(
							$this->createMatrixcodeObject($ordre->TypeCM, array("text" => $this->translate($ordre->NomChamp, true))),
							$ordre->Left*self::TAUX_MM_PT,
							$page->getHeight()-$d-($ordre->Top + $ordre->Width)*self::TAUX_MM_PT,
							$ordre->Width*self::TAUX_MM_PT,
							$ordre->ModuleEntier != "0"
						);
						break;
					case "BarCode":
						$page->drawBarcode(
							$this->createBarcodeObject($ordre->TypeCB, array("text" => $this->translate($ordre->NomChamp), "barHeight" => $ordre->Height*self::TAUX_MM_PT)),
							$ordre->Left*self::TAUX_MM_PT, // x
							$page->getHeight()-$d-($ordre->Top + $ordre->Height)*self::TAUX_MM_PT, // y
							$ordre->Width*self::TAUX_MM_PT, // width
							$ordre->Height*self::TAUX_MM_PT, // height
							//.7, // sizeFactor
							//$ordre->Alignement, // position
							$ordre->Legende * 0.15 // fontFactor
						);
						break;
					case "Line":
						$page->setLineWidth(0.3);
						$page->drawLine($ordre->Left*self::TAUX_MM_PT, $page->getHeight()-$ordre->Top*self::TAUX_MM_PT-$d, ($ordre->Left + $ordre->Width)*self::TAUX_MM_PT, $page->getHeight()-$d-($ordre->Top + $ordre->Height)*self::TAUX_MM_PT);
						break;
					case "Rectangle":
						$page->setLineWidth(0.3);
						$page->setFillColor($this->_colors[$ordre->BackColor % 16777216]);
						$page->setLineColor($this->_colors[$ordre->Couleur % 16777216]);
						$page->drawRectangle($ordre->Left*self::TAUX_MM_PT, $page->getHeight()-$ordre->Top*self::TAUX_MM_PT-$d, ($ordre->Width + $ordre->Left)*self::TAUX_MM_PT, $page->getHeight()-$d-($ordre->Top + $ordre->Height)*self::TAUX_MM_PT);
						break;
					case "Image":
						$page->drawImage($this->_images[(string) $ordre->FichierImage], $ordre->Left*self::TAUX_MM_PT, $page->getHeight() - $d - ($ordre->Top + $ordre->Height)*self::TAUX_MM_PT, ($ordre->Left + $ordre->Width)*self::TAUX_MM_PT, $page->getHeight() - $d - $ordre->Top*self::TAUX_MM_PT);
						break;
					case "TextIfNotNull":
						if ($this->translate(isset($ordre->NomTable)?strtolower($ordre->NomTable).".".$ordre->NomChamp:$ordre->NomChamp) != false)
							$page->drawAlignedText($ordre->Texte, $ordre->Left*self::TAUX_MM_PT, $page->getHeight()-$ordre->Top*self::TAUX_MM_PT-$d-$page->getFontSize(), ($ordre->Left + $ordre->Width)*self::TAUX_MM_PT, $ordre->Alignement, "UTF-8");
						break;
					default:
						$page->drawText("non disponible", $ordre->Left*self::TAUX_MM_PT, $page->getHeight()-$d-$page->getFontSize()-$ordre->Top*self::TAUX_MM_PT, "UTF-8");
						break;
				}
				if ($ordre->Clipped == 1) $page->restoreGS();
			}
		}
	}
	
	/**
	 * Traduit un nom de champ donné en champ de la version Web
	 * @todo alléger le processus après réfection des états
	 * @param string $string le nom du champ
	 * @return string la valeur correspondante
	 */
	private function translate($string, $raw = false) {
		$bits = explode(";",$string);
		
		foreach($bits as $i => $bit) {
			if (substr($bit,0,1) === "#") {
				$bits[$i] = substr($bit, 1, -1);
				continue;
			}
			
			$parts = explode(".", $bit);
			if (count($parts) == 1) {
				try {
					if ($raw)
						$bits[$i] = $this->_objects["default"]->$parts[0];
					else
						$bits[$i] = mb_strtoupper($this->_objects["default"]->$parts[0], "utf-8");
				} catch(Exception $e) {
					$bits[$i] = $parts[0];
				}
			} elseif (count($parts) == 2) {
				if(!array_key_exists($parts[0],$this->_objects)){
					error_log("The value \"$parts[0].$parts[1]\" from state printing model don't exist");
					$bits[$i] = "";
				}else{
					try {
						if ($raw){
							$bits[$i] = $this->_objects[$parts[0]]->$parts[1];
						}
						else {
							$bits[$i] = mb_strtoupper($this->_objects[$parts[0]]->$parts[1], "utf-8");
						}
					} catch(Exception $e) {
						$bits[$i] = $parts[0].".".$parts[1];
					}
				}
			} else {
				$bits[$i] = "dnd";
			}
		}
		return implode("",$bits);
	}
	
	/**
	 * Crée un code barre en fonction du TypeCB fourni
	 * @param string $type
	 * @param array $options les options de l'objet Barcode
	 * @return Zend_Barcode_Object_ObjectAbstract
	 */
	private function createBarcodeObject($type, array $options = null) {
		switch ($type) {
			case "128":
				return new Zend_Barcode_Object_Code128($options);
			case "39":
				return new Zend_Barcode_Object_Code39($options);
		}
	}
	
	private function createMatrixcodeObject($type, array $options = null) {
		switch ($type) {
			case "QrCode":
				return new Application_Plugin_Matrixcode_Qrcode($options);
		}
	}
	
	/**
	 * Formatte la date
	 * @todo à transformer en tableau ?
	 * @param int $format
	 */
	private function formatDate($format) {
		switch ($format) {
			case 2:
				return "d/m/Y H:i:s";
		}
	}
}
